'use client';

import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Progress } from '@/components/ui/progress';
import { Button } from '@/components/ui/button';
import { AlertCircle } from 'lucide-react';

interface UsageStats {
  plan: {
    name: string;
    displayName: string;
  };
  products: {
    used: number;
    limit: number;
    percentage: number;
  };
  categories: {
    used: number;
    limit: number;
    percentage: number;
  };
  aiCredits: {
    used: number;
    limit: number;
    percentage: number;
  };
  languages: {
    limit: number;
  };
  status: string;
  trialEnd: Date | null;
  currentPeriodEnd: Date | null;
  cancelAtPeriodEnd: boolean;
  gracePeriodEnd: Date | null;
}

interface UsageCardProps {
  stats: UsageStats;
  onUpgrade: () => void;
}

export default function UsageCard({ stats, onUpgrade }: UsageCardProps) {
  const isNearLimit = (percentage: number) => percentage >= 80;
  const isAtLimit = (percentage: number) => percentage >= 100;

  const formatDate = (date: Date | null) => {
    if (!date) return null;
    return new Date(date).toLocaleDateString('ro-RO', {
      day: 'numeric',
      month: 'long',
      year: 'numeric',
    });
  };

  return (
    <Card>
      <CardHeader>
        <CardTitle className="flex justify-between items-center">
          <span>Plan: {stats.plan.displayName}</span>
          {stats.plan.name !== 'pro' && stats.plan.name !== 'enterprise' && (
            <Button onClick={onUpgrade} size="sm">
              Upgrade
            </Button>
          )}
        </CardTitle>
      </CardHeader>
      <CardContent className="space-y-6">
        {stats.status === 'trialing' && stats.trialEnd && (
          <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-3 flex items-start gap-2">
            <AlertCircle className="h-5 w-5 text-yellow-600 mt-0.5" />
            <div className="text-sm">
              <p className="font-medium text-yellow-900">Trial activ</p>
              <p className="text-yellow-700">
                Expiră pe {formatDate(stats.trialEnd)}. Alegeți un plan pentru a continua.
              </p>
            </div>
          </div>
        )}

        {stats.status === 'past_due' && stats.gracePeriodEnd && (
          <div className="bg-red-50 border border-red-200 rounded-lg p-3 flex items-start gap-2">
            <AlertCircle className="h-5 w-5 text-red-600 mt-0.5" />
            <div className="text-sm">
              <p className="font-medium text-red-900">Plată eșuată</p>
              <p className="text-red-700">
                Serviciul va fi suspendat pe {formatDate(stats.gracePeriodEnd)} dacă plata nu este finalizată.
              </p>
            </div>
          </div>
        )}

        {stats.cancelAtPeriodEnd && stats.currentPeriodEnd && (
          <div className="bg-orange-50 border border-orange-200 rounded-lg p-3 flex items-start gap-2">
            <AlertCircle className="h-5 w-5 text-orange-600 mt-0.5" />
            <div className="text-sm">
              <p className="font-medium text-orange-900">Abonament anulat</p>
              <p className="text-orange-700">
                Veți reveni la planul Trial pe {formatDate(stats.currentPeriodEnd)}.
              </p>
            </div>
          </div>
        )}

        <div>
          <div className="flex justify-between text-sm mb-2">
            <span className="font-medium">Produse</span>
            <span className={isAtLimit(stats.products.percentage) ? 'text-red-600 font-medium' : ''}>
              {stats.products.used} / {stats.products.limit === 999999 ? '∞' : stats.products.limit}
            </span>
          </div>
          <Progress
            value={Math.min(stats.products.percentage, 100)}
            className={`h-2 ${isAtLimit(stats.products.percentage) ? 'bg-red-200' : isNearLimit(stats.products.percentage) ? 'bg-yellow-200' : ''}`}
          />
          {isNearLimit(stats.products.percentage) && !isAtLimit(stats.products.percentage) && (
            <p className="text-xs text-yellow-600 mt-1">Aproape de limită</p>
          )}
          {isAtLimit(stats.products.percentage) && (
            <p className="text-xs text-red-600 mt-1">Limită atinsă</p>
          )}
        </div>

        <div>
          <div className="flex justify-between text-sm mb-2">
            <span className="font-medium">Categorii</span>
            <span className={isAtLimit(stats.categories.percentage) ? 'text-red-600 font-medium' : ''}>
              {stats.categories.used} / {stats.categories.limit === 999999 ? '∞' : stats.categories.limit}
            </span>
          </div>
          <Progress
            value={Math.min(stats.categories.percentage, 100)}
            className={`h-2 ${isAtLimit(stats.categories.percentage) ? 'bg-red-200' : isNearLimit(stats.categories.percentage) ? 'bg-yellow-200' : ''}`}
          />
        </div>

        {stats.aiCredits.limit > 0 && (
          <div>
            <div className="flex justify-between text-sm mb-2">
              <span className="font-medium">Credite AI (luna aceasta)</span>
              <span className={isAtLimit(stats.aiCredits.percentage) ? 'text-red-600 font-medium' : ''}>
                {stats.aiCredits.used} / {stats.aiCredits.limit}
              </span>
            </div>
            <Progress
              value={Math.min(stats.aiCredits.percentage, 100)}
              className={`h-2 ${isAtLimit(stats.aiCredits.percentage) ? 'bg-red-200' : isNearLimit(stats.aiCredits.percentage) ? 'bg-yellow-200' : ''}`}
            />
          </div>
        )}

        <div className="text-sm">
          <span className="font-medium">Limbi disponibile: </span>
          <span>
            {stats.languages.limit === 1 && 'Română'}
            {stats.languages.limit === 2 && 'Română + Engleză'}
            {stats.languages.limit >= 3 && `${stats.languages.limit} limbi`}
          </span>
        </div>

        {stats.currentPeriodEnd && stats.status === 'active' && !stats.cancelAtPeriodEnd && (
          <div className="text-sm text-gray-600">
            Următoarea facturare: {formatDate(stats.currentPeriodEnd)}
          </div>
        )}
      </CardContent>
    </Card>
  );
}