import { NextRequest, NextResponse } from 'next/server';
import { getServerSession } from 'next-auth';
import { authOptions } from '@/lib/auth';
import { prisma } from '@/lib/prisma';
import { stripe } from '@/lib/stripe';

export async function POST(req: NextRequest) {
  try {
    const session = await getServerSession(authOptions);
    
    if (!session?.user) {
      return NextResponse.json({ error: 'Neautentificat' }, { status: 401 });
    }

    const { planName, billingCycle } = await req.json();

    if (!['basic', 'pro'].includes(planName)) {
      return NextResponse.json({ error: 'Plan invalid' }, { status: 400 });
    }

    if (!['monthly', 'yearly'].includes(billingCycle)) {
      return NextResponse.json({ error: 'Billing cycle invalid' }, { status: 400 });
    }

    const restaurant = await prisma.restaurant.findFirst({
      where: { ownerId: session.user.id },
      include: { subscription: true },
    });

    if (!restaurant) {
      return NextResponse.json({ error: 'Restaurant negăsit' }, { status: 404 });
    }

    const plan = await prisma.plan.findUnique({
      where: { name: planName },
    });

    if (!plan) {
      return NextResponse.json({ error: 'Plan negăsit' }, { status: 404 });
    }

    const priceId = billingCycle === 'monthly' 
      ? plan.stripePriceIdMonthly 
      : plan.stripePriceIdYearly;

    if (!priceId) {
      return NextResponse.json({ error: 'Price ID lipsă pentru acest plan' }, { status: 400 });
    }

    const checkoutSession = await stripe.checkout.sessions.create({
      customer_email: session.user.email!,
      mode: 'subscription',
      line_items: [
        {
          price: priceId,
          quantity: 1,
        },
      ],
      success_url: `${process.env.NEXT_PUBLIC_URL}/dashboard?success=true`,
      cancel_url: `${process.env.NEXT_PUBLIC_URL}/dashboard?canceled=true`,
      metadata: {
        restaurantId: restaurant.id,
        plan: planName,
        billingCycle,
      },
    });

    return NextResponse.json({ sessionUrl: checkoutSession.url });
  } catch (error) {
    console.error('Checkout error:', error);
    return NextResponse.json({ error: 'Eroare la creare checkout' }, { status: 500 });
  }
}