import { NextRequest, NextResponse } from 'next/server';
import { prisma } from '@/lib/prisma';
import bcrypt from 'bcryptjs';
import { registerSchema } from '@/lib/validations';
import { slugify } from '@/lib/utils';
import { addDays } from 'date-fns';

export async function POST(req: NextRequest) {
  try {
    const body = await req.json();
    const { email, password, restaurantName } = registerSchema.parse(body);

    const existingUser = await prisma.user.findUnique({
      where: { email },
    });

    if (existingUser) {
      return NextResponse.json(
        { error: 'Email deja înregistrat' },
        { status: 400 }
      );
    }

    const passwordHash = await bcrypt.hash(password, 10);

    const trialPlan = await prisma.plan.findUnique({
      where: { name: 'trial' },
    });

    if (!trialPlan) {
      return NextResponse.json(
        { error: 'Plan trial negăsit' },
        { status: 500 }
      );
    }

    const result = await prisma.$transaction(async (tx) => {
      const user = await tx.user.create({
        data: {
          email,
          passwordHash,
          emailVerified: new Date(),
        },
      });

      const slug = slugify(restaurantName) + '-' + Math.random().toString(36).substring(7);

      const restaurant = await tx.restaurant.create({
        data: {
          ownerId: user.id,
          name: restaurantName,
          slug,
        },
      });

      const now = new Date();
      const trialEnd = addDays(now, trialPlan.trialDays || 7);

      const subscription = await tx.subscription.create({
        data: {
          restaurantId: restaurant.id,
          planId: trialPlan.id,
          status: 'trialing',
          trialStart: now,
          trialEnd: trialEnd,
        },
      });

      return { user, restaurant, subscription };
    });

    return NextResponse.json({
      message: 'Cont creat cu succes',
      userId: result.user.id,
      restaurantId: result.restaurant.id,
    });
  } catch (error: any) {
    console.error('Register error:', error);
    return NextResponse.json(
      { error: error.message || 'Eroare la înregistrare' },
      { status: 500 }
    );
  }
}