import { PrismaClient } from '@prisma/client';
import bcrypt from 'bcryptjs';
import { addDays } from 'date-fns';

const prisma = new PrismaClient();

async function main() {
  console.log('🌱 Seeding database...');

  const trialPlan = await prisma.plan.findUnique({
    where: { name: 'trial' },
  });

  if (!trialPlan) {
    console.error('❌ Trial plan not found. Run seed-plans.ts first!');
    process.exit(1);
  }

  const adminPassword = await bcrypt.hash('admin123', 10);
  const admin = await prisma.user.upsert({
    where: { email: 'admin@vestho.com' },
    update: {},
    create: {
      email: 'admin@vestho.com',
      passwordHash: adminPassword,
      role: 'super_admin',
      emailVerified: new Date(),
    },
  });

  console.log('✅ Created Super Admin:', admin.email);

  const ownerPassword = await bcrypt.hash('demo123', 10);
  const owner = await prisma.user.upsert({
    where: { email: 'demo@restaurant.ro' },
    update: {},
    create: {
      email: 'demo@restaurant.ro',
      passwordHash: ownerPassword,
      role: 'owner',
      emailVerified: new Date(),
    },
  });

  console.log('✅ Created Demo Owner:', owner.email);

  const restaurant = await prisma.restaurant.upsert({
    where: { slug: 'la-mare' },
    update: {},
    create: {
      ownerId: owner.id,
      name: 'Restaurant La Mare',
      slug: 'la-mare',
      address: 'Strada Mării 123, București',
      phone: '+40 21 123 4567',
      primaryLanguage: 'ro',
      status: 'active',
    },
  });

  console.log('✅ Created Restaurant:', restaurant.name);

  const now = new Date();
  const trialEnd = addDays(now, trialPlan.trialDays || 7);

  const subscription = await prisma.subscription.upsert({
    where: { restaurantId: restaurant.id },
    update: {},
    create: {
      restaurantId: restaurant.id,
      planId: trialPlan.id,
      status: 'trialing',
      trialStart: now,
      trialEnd: trialEnd,
    },
  });

  console.log('✅ Created Subscription:', subscription.status);

  const categoriesData = [
    { nameRo: 'Aperitive', nameEn: 'Appetizers', icon: '🥗', sortOrder: 1 },
    { nameRo: 'Feluri Principale', nameEn: 'Main Courses', icon: '🍽️', sortOrder: 2 },
    { nameRo: 'Desert', nameEn: 'Desserts', icon: '🍰', sortOrder: 3 },
  ];

  const categories = [];
  for (const cat of categoriesData) {
    const category = await prisma.category.create({
      data: {
        restaurantId: restaurant.id,
        ...cat,
      },
    });
    categories.push(category);
  }

  console.log('✅ Created Categories:', categories.length);

  const productsData = [
    {
      categoryId: categories[0].id,
      nameRo: 'Salată Caesar',
      nameEn: 'Caesar Salad',
      descriptionRo: 'Salată verde, piept de pui, crutoane, parmezan, sos Caesar',
      descriptionEn: 'Green salad, chicken breast, croutons, parmesan, Caesar dressing',
      price: 35.00,
      allergens: ['gluten', 'lactate', 'ouă'],
      sortOrder: 1,
    },
    {
      categoryId: categories[1].id,
      nameRo: 'Paste Carbonara',
      nameEn: 'Pasta Carbonara',
      descriptionRo: 'Paste cu bacon, ou, parmezan și piper negru',
      descriptionEn: 'Pasta with bacon, egg, parmesan and black pepper',
      price: 45.00,
      allergens: ['gluten', 'lactate', 'ouă'],
      sortOrder: 1,
    },
    {
      categoryId: categories[2].id,
      nameRo: 'Tiramisu',
      nameEn: 'Tiramisu',
      descriptionRo: 'Desert tradițional italian cu mascarpone și cafea',
      descriptionEn: 'Traditional Italian dessert with mascarpone and coffee',
      price: 32.00,
      allergens: ['gluten', 'lactate', 'ouă'],
      sortOrder: 1,
    },
  ];

  for (const prod of productsData) {
    await prisma.product.create({
      data: {
        restaurantId: restaurant.id,
        ...prod,
      },
    });
  }

  console.log('✅ Created Products:', productsData.length);
  console.log('🎉 Seeding completed!');
}

main()
  .catch((e) => {
    console.error('❌ Seeding error:', e);
    process.exit(1);
  })
  .finally(async () => {
    await prisma.$disconnect();
  });